NodeManager

# Introduction

MySensors (<https://www.mysensors.org>) is an open source hardware and software community focusing on do-it-yourself home automation and Internet of Things which allows creating original and affordable sensors.

NodeManager is intended to take care on your behalf of all those common tasks a MySensors node has to accomplish, speeding up the development cycle of your projects.

NodeManager includes the following main components:

* Sleep manager: allows managing automatically the complexity behind battery-powered sensors spending most of their time sleeping
* Power manager: allows powering on your sensors only while the node is awake
* Battery manager: provides common functionalities to read and report the battery level
* Remote configuration: allows configuring remotely the node without the need to have physical access to it
* Built-in sensors: for the most common sensors, provide embedded code so to allow their configuration with a single line

## Features

* Manage all the aspects of a sleeping cycle by leveraging smart sleep
* Allow configuring the sleep mode and the sleep duration remotely
* Allow waking up a sleeping node remotely at the end of a sleeping cycle
* Allow powering on each connected sensor only while the node is awake to save battery
* Report battery level periodically and automatically
* Calculate battery level without requiring an additional pin and the resistors
* Report battery voltage through a built-in sensor
* Can report battery level on demand
* Allow rebooting the board remotely
* Provide out-of-the-box sensors personalities and automatically execute their main task at each cycle

# Installation
* Download the package from https://mynodemanager.sourceforge.io
* Open the provided sketch template and save it under a different name
* Open `config.h` and customize both MySensors configuration and NodeManager global settings
* Register your sensors in the sketch file
* Upload the sketch to your arduino board

Please note NodeManager cannot be used as an arduino library since requires access to your MySensors configuration directives, hence its files have to be placed into the same directory of your sketch.

# Configuration
NodeManager configuration includes compile-time configuration directives (which can be set in config.h), runtime global and per-sensor configuration settings (which can be set in your sketch) and settings that can be customized remotely (via a special child id).

## Setup MySensors
Since NodeManager has to communicate with the MySensors gateway on your behalf, it has to know how to do it. Place on top of the `config.h` file all the MySensors typical directives you are used to set on top of your sketch so both your sketch AND NodeManager will be able to share the same configuration.

## Enable/Disable NodeManager's modules

Those NodeManager's directives in the `config.h` file control which module/library/functionality will be made available to your sketch. Enable (e.g. set to 1) only what you need to ensure enough space is left to your custom code.

~~~c
// if enabled, will load the sleep manager library. Sleep mode and sleep interval have to be configured to make the board sleeping/waiting
#define SLEEP_MANAGER 1
// if enabled, enable the capability to power on sensors with the arduino's pins to save battery while sleeping
#define POWER_MANAGER 1
// if enabled, will load the battery manager library to allow the battery level to be reported automatically or on demand
#define BATTERY_MANAGER 1
// if enabled, allow modifying the configuration remotely by interacting with the configuration child id
#define REMOTE_CONFIGURATION 1
// if enabled, persist the configuration settings on EEPROM
#define PERSIST 0

// if enabled, enable debug messages on serial port
#define DEBUG 1

// if enabled, send a SLEEPING and AWAKE service messages just before entering and just after leaving a sleep cycle
#define SERVICE_MESSAGES 1
// if enabled, a battery sensor will be created at BATTERY_CHILD_ID and will report vcc voltage together with the battery level percentage
#define BATTERY_SENSOR 1


// Enable this module to use one of the following sensors: SENSOR_ANALOG_INPUT, SENSOR_LDR, SENSOR_THERMISTOR
#define MODULE_ANALOG_INPUT 1
// Enable this module to use one of the following sensors: SENSOR_DIGITAL_INPUT
#define MODULE_DIGITAL_INPUT 1
// Enable this module to use one of the following sensors: SENSOR_DIGITAL_OUTPUT, SENSOR_RELAY, SENSOR_LATCHING_RELAY
#define MODULE_DIGITAL_OUTPUT 1
// Enable this module to use one of the following sensors: SENSOR_SHT21
#define MODULE_SHT21 0
// Enable this module to use one of the following sensors: SENSOR_DHT11, SENSOR_DHT22
#define MODULE_DHT 0
// Enable this module to use one of the following sensors: SENSOR_SWITCH, SENSOR_DOOR, SENSOR_MOTION
#define MODULE_SWITCH 0
// Enable this module to use one of the following sensors: SENSOR_DS18B20
#define MODULE_DS18B20 0
~~~

## Configure NodeManager

Node Manager comes with a reasonable default configuration. If you want/need to change its settings, this can be done in your sketch, inside the `before()` function and just before registering your sensors. The following methods are exposed for your convenience:

~~~c
    // the pin to connect to the RST pin to reboot the board (default: 4)
    void setRebootPin(int value);
    #if BATTERY_MANAGER == 1
      // the expected vcc when the batter is fully discharged, used to calculate the percentage (default: 2.7)
      void setBatteryMin(float value);
      // the expected vcc when the batter is fully charged, used to calculate the percentage (default: 3.3)
      void setBatteryMax(float value);
      // how frequently (in hours) to report the battery level to the controller. When reset the battery is always reported (default: 1)
      void setBatteryReportCycles(int value);
    #endif
    #if SLEEP_MANAGER == 1
      // define if the board has to sleep every time entering loop (default: IDLE). It can be IDLE (no sleep), SLEEP (sleep at every cycle), WAIT (wait at every cycle
      void setSleepMode(int value);
      // define for how long the board will sleep (default: 0)
      void setSleepTime(int value);
      // define the unit of SLEEP_TIME. It can be SECONDS, MINUTES, HOURS or DAYS (default: MINUTES)
      void setSleep(int value1, int value2, int value3);
      void setSleepUnit(int value);
      // if enabled, when waking up from the interrupt, the board stops sleeping. Disable it when attaching e.g. a motion sensor (default: true)
      void setSleepInterruptPin(int value);
    #endif
    // configure the interrupt pin and mode. Mode can be CHANGE, RISING, FALLING (default: MODE_NOT_DEFINED)
    void setInterrupt(int pin, int mode, int pull = -1);
    // register a built-in sensor
    int registerSensor(int sensor_type, int pin = -1, int child_id = -1);
    // register a custom sensor
    int registerSensor(Sensor* sensor);
    // return a sensor by its index
    Sensor* get(int sensor_index);
    #if POWER_MANAGER == 1
      // to save battery the sensor can be optionally connected to two pins which will act as vcc and ground and activated on demand
      void setPowerPins(int ground_pin, int vcc_pin, long wait = 10);
      void powerOn();
      void powerOff();
    #endif
~~~

For example

~~~c
nodeManager.setBatteryMin(1.8);
~~~

## Register your sensors
In your sketch, inside the `before()` function and just before calling `nodeManager.before()`, you can register your sensors against NodeManager. The following built-in sensor types are available:

Sensor type  | Description
 ------------- | -------------
SENSOR_ANALOG_INPUT | Generic analog sensor, return a pin's analog value or its percentage
SENSOR_LDR | LDR sensor, return the light level of an attached light resistor in percentage
SENSOR_THERMISTOR | Thermistor sensor, return the temperature based on the attached thermistor
SENSOR_DIGITAL_INPUT |  Generic digital sensor, return a pin's digital value
SENSOR_DIGITAL_OUTPUT | Generic digital output sensor, allows setting the digital output of a pin to the requested value
SENSOR_RELAY | Relay sensor, allows activating the relay
SENSOR_LATCHING_RELAY| Latching Relay sensor, allows activating the relay with a pulse
SENSOR_DHT11 | DHT11 sensor, return temperature/humidity based on the attached DHT sensor
SENSOR_DHT22 | DHT22 sensor, return temperature/humidity based on the attached DHT sensor
SENSOR_SHT21 | SHT21 sensor, return temperature/humidity based on the attached SHT21 sensor
SENSOR_SWITCH | Generic switch, wake up the board when a pin changes status
SENSOR_DOOR | Door sensor, wake up the board and report when an attached magnetic sensor has been opened/closed
SENSOR_MOTION | Motion sensor, wake up the board and report when an attached PIR has triggered
SENSOR_DS18B20 | DS18B20 sensor, return the temperature based on the attached sensor

To register a sensor simply call the NodeManager instance with the sensory type and the pin the sensor is conncted to. For example:
~~~c
nodeManager.registerSensor(SENSOR_THERMISTOR,A2);
nodeManager.registerSensor(SENSOR_DOOR,3);
~~~

Once registered, your job is done. NodeManager will assign a child id automatically, present each sensor for you to the controller, query each sensor and report the value back to the gateway/controller at at the end of each sleep cycle . For actuators (e.g. relays) those can be triggered by sending a `REQ` message to their assigned child id.

When called, registerSensor returns the child_id of the sensor so you will be able to retrieve it later if needed. If you want to set a child_id manually, this can be passed as third argument to the function.

### Creating a custom sensor

If you want to create a custom sensor and register it with NodeManager so it can take care of all the common tasks, you can create a class inheriting from `Sensor` and implement the following methods:
~~~c
    // define what to do during before() to setup the sensor
    void onBefore();
    // define what to do during loop() by executing the sensor's main task
    void onLoop();
    // define what to do during receive() when the sensor receives a message
    void onReceive(const MyMessage & message);
~~~

You can then instantiate your newly created class and register with NodeManager:
~~~c
nodeManager.registerSensor(new SensorCustom(child_id, pin));
~~~

## Configuring the sensors
Each built-in sensor class comes with reasonable default settings. In case you want/need to customize any of those settings, after having registered the sensor, you can retrieve it back and call set functions common to all the sensors or specific for a given class.

To do so, use `nodeManager.get(child_id)` which will return a pointer to the sensor. Remeber to cast it to the right class before calling their functions. For example:

~~~c
((SensorLatchingRelay*)nodeManager.get(2))->setPulseWidth(50);
~~~


### Sensor's general configuration

The following methods are available for all the sensors:
~~~c
    // where the sensor is attached to (default: not set)
    void setPin(int value);
    // child_id of this sensor (default: not set)
    void setChildId(int value);
    // presentation of this sensor (default: S_CUSTOM)
    void setPresentation(int value);
    // type of this sensor (default: V_CUSTOM)
    void setType(int value);
    // when queried, send the message multiple times (default: 1)
    void setRetries(int value);
    // For some sensors, the measurement can be queried multiple times and an average is returned (default: 1)
    void setSamples(int value);
    // If more then one sample has to be taken, set the interval in milliseconds between measurements (default: 0)
    void setSamplesInterval(int value);
    // if true will report the measure only if different then the previous one (default: false)
    void setTackLastValue(bool value);
    // if track last value is enabled, force to send an update after the configured number of cycles (default: -1)
    void setForceUpdate(int value);
    // the value type of this sensor (default: TYPE_INTEGER)
    void setValueType(int value);
// for float values, set the float precision (default: 2)
    void setFloatPrecision(int value);
    #if POWER_MANAGER == 1
      // to save battery the sensor can be optionally connected to two pins which will act as vcc and ground and activated on demand
      void setPowerPins(int ground_pin, int vcc_pin, long wait = 0);
      void powerOn();
      void powerOff();
    #endif
~~~

### Sensor's specific configuration

Each sensor class can expose additional methods.

#### SensorAnalogInput / SensorLDR
~~~c
    // the analog reference to use (default: not set, can be either INTERNAL or DEFAULT)
    void setReference(int value);
    // reverse the value or the percentage (e.g. 70% -> 30%) (default: false)
    void setReverse(bool value);
    // when true returns the value as a percentage (default: true)
    void setOutputPercentage(bool value);
    // minimum value for calculating the percentage (default: 0)
    void setRangeMin(int value);
    // maximum value for calculating the percentage (default: 1024)
    void setRangeMax(int value);
~~~

#### SensorThermistor
~~~c
    // resistance at 25 degrees C (default: 10000)
    void setNominalResistor(int value);
    // temperature for nominal resistance (default: 25)
    void setNominalTemperature(int value);
    // The beta coefficient of the thermistor (default: 3950)
    void setBCoefficient(int value);
    // the value of the resistor in series with the thermistor (default: 10000)
    void setSeriesResistor(int value);
    // set a temperature offset
    void setOffset(float value);
~~~

#### SensorDigitalOutput / SensorRelay / SensorLatchingRelay
~~~c
    // set how to initialize the output (default: LOW)
    void setInitialValue(int value);
    // if greater than 0, send a pulse of the given duration in ms and then restore the output back to the original value (default: 0)
    void setPulseWidth(int value);
~~~

#### SensorSwitch / SensorDoor / SensorMotion
~~~c
    // set the interrupt mode. Can be CHANGE, RISING, FALLING (default: CHANGE)
    void setMode(int value);
    // milliseconds to wait before reading the input (default: 0)
    void setDebounce(int value);
    // time to wait in milliseconds after a change is detected to allow the signal to be restored to its normal value (default: 0)
    void setTriggerTime(int value);
~~~

## Upload your sketch

Upload your sketch to your arduino board as you are used to.

## Verify if everything works fine

Check your gateway's logs to ensure the node is working as expected. You should see the node presenting itself, reporting battery level, presenting all the registered sensors and the configuration child id service.
When `DEBUG` is enabled, detailed information is available through the serial port. Remember to disable debug once the tests have been completed.

## Communicate with each sensor

You can interact with each registered sensor asking to execute their main tasks by sending to the child id a `REQ` command. For example to request the temperature to node_id 254 and child_id 1:

`254;1;2;0;0;`

To activate a relay connected to the same node, child_id 100:

`254;100;2;0;2;1`

No need to implement anything on your side since for built-in sensor types this is handled automatically. 
Once the node will be sleeping, it will report automatically each measure at the end of every sleep cycle.

## Communicate with the node

NodeManager exposes a configuration service by default on child_id 200 so you can interact with it by sending `V_CUSTOM` type of messages and commands within the payload. For each `REQ` message, the node will respond with a `SET` message.
The following custom commands are available:

NodeManager command  | Description
 ------------- | -------------
BATTERY | Report the battery level back to the gateway/controller
HELLO | Hello request
REBOOT | Reboot the board
CLEAR | Wipe from the EEPROM NodeManager's settings
VERSION | Respond with NodeManager's version
IDxxx |  Change the node id to the provided one. E.g. ID025: change the node id to 25. Requires a reboot to take effect
INTVLnnnX | Set the wait/sleep interval to nnn where X is S=Seconds, M=mins, H=Hours, D=Days. E.g. INTVL010M would be 10 minutes
MODEx | Change the way the board behaves  (e.g. MODE1). 0: stay awake, 1: go to sleep for the configured interval, 2: wait for the configured interval
AWAKE | When received after a sleeping cycle or during wait, abort the cycle and stay awake

For example, to request the battery level to node id 254:

`254;200;2;0;48;BATTERY`

To set the sleeping cycle to 1 hour:

`254;200;2;0;48;INTVL001H`

To ask the node to start sleeping (and waking up based on the previously configured interval):

`254;200;2;0;48;MODE1`

To wake up a node previously configured with `MODE1`, send the following just after reporting `AWAKE`:

`254;200;2;0;48;WAKEUP`

In addition, NodeManager will report with custom messages every time the board is going to sleep (`SLEEPING`) or it is awake (`AWAKE`).

If `PERSIST` is enabled, the settings provided with `INTVLnnnX` and `MODEx` are saved to the EEPROM to be persistent even after rebooting the board.

# How it works

A NodeManager object must be created and called from within your sketch during `before()`, `presentation()`, `loop()` and `receive()` to work properly. NodeManager will do the following during each phase:

## NodeManager::before()
* Configure the reboot pin so to allow rebooting the board
* Setup the interrupt pins to wake up the board based on the configured interrupts (e.g. stop sleeping when the pin is connected to ground or wake up and notify when a motion sensor has trigger)
* If persistance is enabled, restore from the EEPROM the latest sleeping settings
* Call `before()` of each registered sensor

### Sensor::before()
* Call sensor-specific implementation of before by invoking `onBefore()` to initialize the sensor

## NodeManager::loop()
* If all the sensors are powered by an arduino pin, this is set to HIGH
* Call `loop()` of each registered sensor
* If all the sensors are powered by an arduino pin, this is set to LOW

### Sensor::loop()
* If the sensor is powered by an arduino pin, this is set to HIGH
* For each registered sensor, the sensor-specific `onLoop()` is called. If multiple samples are requested, this is run multiple times.
* In case multiple samples have been collected, the average is calculated
* A message is sent to the gateway with the calculated value. Depending on the configuration, this is not sent if it is the same as the previous value or sent anyway after a given number of cycles. These functionalies are not sensor-specific and common to all the sensors inheriting from the `Sensor` class.
* If the sensor is powered by an arduino pin, this is set to LOW

## NodeManager::receive()
* Receive a message from the radio network 
* If the destination child id is the configuration node, it will handle the incoming message, otherwise will dispatch the message to the recipient sensor

### Sensor::receive()
* Invoke `Sensor::loop()` which will execute the sensor main taks and eventually call `Sensor::onReceive()`