#############################################################################m#
#  Skript to generate Austrian experience mortality table objects          ####
#############################################################################m#
library(MortalityTables)
library(here)
library(readxl)
library(tidyverse)




############################################################################### #
# Bestandssterbetafel der Versucherungsunternehmen Österreichs, 2012-16      ----
############################################################################### #

VUGBfile = here::here("data-raw", "AT", "Experience", "2012-2016_VU-Gesamtbestand", "Gesamtbestandstafel.xlsx")
VUGB.detailfile = here::here("data-raw", "AT", "Experience", "2012-2016_VU-Gesamtbestand", "Gesamtbestandstafel_detail.xlsx")
VUGB.stornofile = here::here("data-raw", "AT", "Experience", "2012-2016_VU-Gesamtbestand", "Stornovektor_detail.xlsx")


#----------------------------------------------------#
## Details zu Sterblichkeiten und Storni          ####
#----------------------------------------------------#


read_VUGB = function(file, sheet, ...) {
  # Prepare column names for the actual data columns
  headers = readxl::read_excel(file, sheet = sheet, skip = 3,col_names = FALSE, n_max = 5)
  hh = headers %>%
    t %>%
    `colnames<-`(c("Tarif", "Wahrscheinlichkeit", "Prämie", "Geschlecht", "Typ")) %>%
    as_tibble()  %>%
    mutate(
      Tarif = recode(Tarif, "Alle Tarife" = "Alle"),
      Wahrscheinlichkeit = recode(Wahrscheinlichkeit, "Sterblichkeiten" = "qx", "Exposure" = NA_character_, "Stornowahrscheinlichkeit" = "sx"),
      Prämie = recode(Prämie, "Alle Zahlarten" = "Alle", "laufende Prämienzahlung" = "laufend", "Einmalerlag" = "Einmalerlag", "Praemienfrei" = "prämienfrei", "Prämienfrei" = "prämienfrei"),
      Geschlecht = recode(Geschlecht, "UNISEX" = "u", "Männer" = "m", "Frauen" = "w"),
      Typ = recode(Typ,
                   "qx roh" = "raw",
                   "qy roh" = "raw",
                   "qu roh" = "raw",
                   "qx" = "smooth",
                   "qy" = "smooth",
                   "qu" = "smooth",
                   "sx" = "raw",
                   "sy" = "raw",
                   "su" = "raw")) %>%
    fill(everything()) %>%
    mutate(varname = paste(Tarif, Wahrscheinlichkeit, Prämie, Geschlecht, Typ, sep = "."))

  headers = hh$varname
  headers[[1]] = "Alter"

  # Read in the actual data
  readxl::read_excel(file, sheet = sheet, skip = 8, col_names = headers) %>%
    gather("key", "Value", -Alter) %>%
    separate(col = "key", into = c("Tarif", "Wahrscheinlichkeit", "Prämie", "Geschlecht", "Typ"), sep = "\\.", remove = TRUE) %>%
    select("tariff" = "Tarif", "sex" = "Geschlecht", premium = Prämie, probability = Wahrscheinlichkeit, age = Alter, Typ, Value)
}

VUBestandstafel.data = bind_rows(
  read_VUGB(file = VUGB.detailfile, sheet = "Gesamtbestand"),
  read_VUGB(file = VUGB.detailfile, sheet = "KLV"),
  read_VUGB(file = VUGB.detailfile, sheet = "FLV"),
  read_VUGB(file = VUGB.detailfile, sheet = "Sonstige"),

  read_VUGB(file = VUGB.stornofile, sheet = "Stornovektor"),
  read_VUGB(file = VUGB.stornofile, sheet = "KLV"),
  read_VUGB(file = VUGB.stornofile, sheet = "FLV"),
  read_VUGB(file = VUGB.stornofile, sheet = "Sonstige")
) %>%
  acast(tariff ~ sex ~ premium ~ probability ~ age ~ Typ, value.var = "Value")


dmn = dimnames(VUBestandstafel.data)[1:4]
# Re-Order the dimension to the "intuitive" order:
dmn[[1]] = dmn[[1]][c(1,3,2,4)]
dmn[[2]] = dmn[[2]][c(2,1,3)]
dmn[[3]] = dmn[[3]][c(1,3,2,4)]

VU.Gesamtbestand.Detail = array(
  data = c(mortalityTable.NA),
  dim = c(dim(VUBestandstafel.data)[1:4], 2),
  dimnames = c(`names<-`(dmn, c("Tarif", "Geschlecht", "Prämie", "Wahrscheinlichkeit")), list(Typ = c("raw", "smooth")))
)

q = "qx"
ages = 0:99
ages.ch = as.character(ages)
for (t in dmn[[1]]) {
  for (s in dmn[[2]]) {
    for (p in dmn[[3]]) {
      VU.Gesamtbestand.Detail[[t, s, p, q, "smooth"]] = mortalityTable.period(
        name = sprintf("VU Gesamtbestand AT, %s, %s, %s, %s, geglättet", t, s, p, q),
        ages = ages,
        deathProbs = VUBestandstafel.data[t, s, p, q, ages.ch, "smooth"],
        exposures = VUBestandstafel.data[t, s, p, q, ages.ch, "Exposure"],
        baseYear = 2014,
        data = list(
          raw = VUBestandstafel.data[t, s, p, q, ages.ch, "raw"],
          dim = list(
            sex = s,
            year = "2012-2016",
            data = "smooth",
            tarif = t,
            table = "VU Gesamtbestand AT",
            zahlart = p,
            probability = q
          ))
      )
      VU.Gesamtbestand.Detail[[t, s, p, q, "raw"]] = mortalityTable.period(
        name = sprintf("VU Gesamtbestand AT, %s, %s, %s, %s, roh", t, s, p, q),
        ages = ages,
        deathProbs = VUBestandstafel.data[t, s, p, q, ages.ch, "raw"],
        exposures = VUBestandstafel.data[t, s, p, q, ages.ch, "Exposure"],
        baseYear = 2014,
        data = list(dim = list(
          sex = s,
          year = "2012-2016",
          data = "raw",
          tarif = t,
          table = "VU Gesamtbestand AT",
          zahlart = p,
          probability = q
        ))
      )
    }
  }
}
q = "sx"
ages = 0:40
ages.ch = as.character(ages)
for (t in dmn[[1]]) {
  for (s in dmn[[2]]) {
    for (p in dmn[[3]]) {
      VU.Gesamtbestand.Detail[[t, s, p, q, "raw"]] = mortalityTable.period(
        name = sprintf("VU Gesamtbestand AT, %s, %s, %s, %s, roh", t, s, p, q),
        ages = ages,
        deathProbs = VUBestandstafel.data[t, s, p, q, ages.ch, "raw"],
        exposures = VUBestandstafel.data[t, s, p, q, ages.ch, "Exposure"],
        baseYear = 2014,
        data = list(dim = list(
          sex = s,
          year = "2012-2016",
          data = "raw",
          tarif = t,
          table = "VU Gesamtbestand AT",
          zahlart = p,
          probability = q
        ))
      )
    }
  }
}

save(VU.Gesamtbestand.Detail, file = here::here("data", "VU.Gesamtbestand.Detail.RData"))



#----------------------------------------------------#
## Gesamtbestandssterblichkeit                    ####
#----------------------------------------------------#
sex = c("m", "w", "u")

VU.Gesamtbestand = array(
  data = c(mortalityTable.NA),
  dim = c(3),
  dimnames = list(sex = sex)
)

ages = as.integer(dimnames(VUBestandstafel.data)[[5]])

for (s in sex) {
  VU.Gesamtbestand[[s]] = mortalityTable.period(
    name = paste0("VU-Gesamtbestand 2012-2016, ", recode(s, "m" = "Männer", "f" = "Frauen", "u" = "Unisex")),
    ages = ages,
    deathProbs = VUBestandstafel.data["Alle", s, "laufend", "qx",,"smooth"],
    exposures = VUBestandstafel.data["Alle", s, "laufend", "qx",,"Exposure"],
    baseYear = 2014,
    data = list(
      dim = list(
        sex = s,
        year = "2012-2016",
        data = "smooth",
        tarif = "Gesamtbestand",
        table = "VU Gesamtbestand AT",
        probability = "qx",
        population = "Lebensversicherte",
        country = "AT",
        period = "2012-2016"
      )
    )
  )
}
save(VU.Gesamtbestand, file = here::here("data", "VU.Gesamtbestand.RData"))



#----------------------------------------------------#
## Gesamtbestandsstornovektor                     ####
#----------------------------------------------------#
tarif = c("KLV", "FLV", "Sonstige")

VU.Gesamtbestand.Storno = array(
  data = c(mortalityTable.NA),
  dim = c(3),
  dimnames = list(tarif = tarif)
)

lz = 0:40
for (t in tarif) {
  tt = recode(t, "Sonstige" = "Sonstige Tarife")
  VU.Gesamtbestand.Storno[[t]] = mortalityTable.period(
    name = paste0("VU-Gesamtbestand 2012-2016, Storno ", t),
    ages = lz,
    deathProbs = VUBestandstafel.data[tt, "u", "laufend", "qx", as.character(lz),"raw"],
    exposures = VUBestandstafel.data[tt, "u", "laufend", "qx", as.character(lz),"Exposure"],
    baseYear = 2014,
    data = list(
      dim = list(
        sex = "u",
        year = "2014",
        data = "raw",
        tarif = t,
        table = "VU Gesamtbestand AT",
        probability = "sx",
        population = "Lebensversicherte",
        country = "AT",
        period = "2012-2016"
      )
    )
  )
}
save(VU.Gesamtbestand.Storno, file = here::here("data", "VU.Gesamtbestand.Storno.RData"))







############################################################################### #
# Bestandssterbetafel der Pensionskassen Österreichs, 2010-16                ----
############################################################################### #

PKfile = here::here("data-raw", "AT", "Experience", "2018-02-15_Auswertung_Bestandsabfrage_Pensionskassen.xlsx")

PKdata = read_excel(PKfile, sheet = "Gesamtbestand",
  skip = 7,
  col_names = c(
    "Alter",
    "m.AWB.BoY", "m.AWB.Tod", "m.AWB.Sonstige", "m.AWB.EoY",
    "m.EP.BoY",  "m.EP.Tod",  "m.EP.Sonstige",  "m.EP.EoY",
    "m.WIP.BoY", "m.WIP.Tod", "m.WIP.Sonstige", "m.WIP.EoY",
    "w.AWB.BoY", "w.AWB.Tod", "w.AWB.Sonstige", "w.AWB.EoY",
    "w.EP.BoY",  "w.EP.Tod",  "w.EP.Sonstige",  "w.EP.EoY",
    "w.WIP.BoY", "w.WIP.Tod", "w.WIP.Sonstige", "w.WIP.EoY"
  )
)


PKdata.array = PKdata %>%
  gather(Variable, Value, -Alter) %>%
  separate(Variable, into = c("sex", "collar", "Variable")) %>%
  # Add unisex (sum of m + w)
  spread(sex, Value) %>%
  mutate(u = m + w) %>%
  gather(sex, Value, m, w, u) %>%
  mutate(sex = factor(sex, levels = c("m", "w", "u"))) %>%
  # Add Gesamt (sum of AWB + EP + WIP)
  spread(collar, Value) %>%
  mutate(Gesamt = AWB + EP + WIP) %>%
  gather(collar, Value, AWB, EP, WIP, Gesamt) %>%
  # Calculate exposure and raw probabilities
  spread(Variable, Value) %>%
  mutate(Exposure = BoY,
         Exposure.qx = Exposure - Sonstige/2,
         Exposure.sx = Exposure - Tod/2,
         qx = ifelse(Exposure.qx == 0, 0,Tod / Exposure.qx), sx = ifelse(Exposure.sx == 0, 0, Sonstige / Exposure.sx)) %>%
  mutate(BoY = NULL, EoY = NULL) %>%
  # Create an array with dimensions Alter - Sex - Bestand - Varible
  gather(Variable, Value, Exposure, Exposure.qx, Exposure.sx, Sonstige, Tod, qx, sx) %>%
  acast(sex ~ collar ~ Alter ~ Variable, value.var = "Value")


# Store all tables in a huge array. Pre-create the array here with NA mortality tables
PKBestandstafel.2010.16    = array(
  data = c(mortalityTable.NA),
  # Use sex and collar from the array, add probability (qx/sx) and type (raw, smoothed) as new dimensions
  dim = c(dim(PKdata.array)[1:2], 2, 2),
  dimnames = c(dimnames(PKdata.array)[1:2], list(probability = c("qx", "sx"), data = c("raw", "smoothed")))
)

ages = as.integer(dimnames(PKdata.array)[[3]])
for (sx in dimnames(PKdata.array)[[1]]) {
  for (cl in dimnames(PKdata.array)[[2]]) {
    sex = switch(sx, "m" = "Männer", "w" = "Frauen", "u" = "Unisex")
    pension = switch(cl,
                     "Gesamt" = "Gesamtbestand",
                     "AWB" = "Anwartschaftsberechtigte",
                     "EP" = "Eigenpension",
                     "WIP" = "Witwenpension"
    )

    tablename = sprintf("PK-Bestand 2010-16, qx %s %s", cl, sex)
    ages.used = PKdata.array[sx,cl,,"Exposure"] > 0
    PKBestandstafel.2010.16[[sx, cl, "qx", "raw"]] = mortalityTable.period(
      name = tablename,
      deathProbs = PKdata.array[sx,cl,ages.used,"qx"],
      ages = ages[ages.used],
      baseYear = 2013,
      exposures = PKdata.array[sx,cl, ages.used,"Exposure"],
      data = list(
        dim = list(sex = sx, year = "2010-2016", data = "raw", collar = cl, type = cl, Tafel = "PK-Bestandsabfrage 2010-16", risk = "Tod", probability = "qx", table = tablename),
        deaths = PKdata.array[sx, cl, ages.used, "Tod"]
      )
    )
    PKBestandstafel.2010.16[[sx, cl, "qx", "smoothed"]] = PKBestandstafel.2010.16[[sx, cl, "qx", "raw"]] %>%
      whittaker.mortalityTable(lambda = 1/2, d = 2, name.postfix = ", geglättet") %>%
      mT.setDimInfo(data = "geglättet")

    tablename = sprintf("PK-Bestand 2010-16, sx %s %s", cl, sex)
    PKBestandstafel.2010.16[[sx, cl, "sx", "raw"]] = mortalityTable.period(
      name = tablename,
      deathProbs = PKdata.array[sx,cl,ages.used,"sx"],
      ages = ages[ages.used],
      baseYear = 2013,
      exposures = PKdata.array[sx,cl,ages.used,"Exposure"],
      data = list(
        dim = list(sex = sx, year = "2010-2016", data = "raw", collar = cl, type = cl, Tafel = "PK-Bestandsabfrage 2010-16", risk = "sonst. Ausscheiden", probability = "sx", table = tablename),
        deaths = PKdata.array[sx, cl, ages.used, "Sonstige"]
      )
    )
    PKBestandstafel.2010.16[[sx, cl, "sx", "smoothed"]] = PKBestandstafel.2010.16[[sx, cl, "sx", "raw"]] %>%
      whittaker.mortalityTable(lambda = 1/10, d = 2, name.postfix = ", geglättet") %>%
      mT.setDimInfo(data = "geglättet")
    }
}

save(PKBestandstafel.2010.16, file = here::here("data", "PKBestandstafel.2010.16.RData"))


# plotMortalityTables(PKBestandstafel.2010.16, legend.position = "bottom")
#
# plotMortalityTables(PKBestandstafel.2010.16["m",,"qx",], aes = aes(linetype = data, color = table), ages = 25:99, legend.position = "bottom")
# plotMortalityTables(PKBestandstafel.2010.16["u",,"sx",], aes = aes(linetype = data, color = collar), ages = 25:99, legend.position = "bottom") + labs(color = NULL, linetype = NULL)
# plotMortalityTables(PKBestandstafel.2010.16["u",,"qx",], aes = aes(linetype = data, color = collar), ages = 25:99, legend.position = "bottom") + labs(color = NULL, linetype = NULL)
