#' @import MortalityTables
NULL

stopifnot(require(methods), require(utils), require(MortalityTables)) # MortalityTable classes; new; Excel reader

############################################################################### #
# Heubeck Table DAV 2005-G (Male, Female) ----
############################################################################### #


#---------------------------------------------------------------------------------------------- -
# Functions and classes to handle Heubeck's projection scales for individual probabilities ----
#---------------------------------------------------------------------------------------------- -
projection.Heubeck05 = function(object, projectionPeriod = 0) {
  ages = ages(object)

  # First 10 years use the short-term trend factor, for all other years the long-term trend factor is used
  # Projection in the past works similar
  shortTrendExponent = sign(projectionPeriod) * pmin(abs(projectionPeriod), 10)
  longTrendExponent = sign(projectionPeriod) * pmax(abs(projectionPeriod) - 10, 0)

  ((1 - object@trend)^(shortTrendExponent)) * ((1 - object@trend2)^(longTrendExponent))
}


projection.Heubeck05.qi = function(object, projectionPeriod = 0) {
  ages = ages(object)

  # First 10 years use the short-term trend factor, for all other years the long-term trend factor is used
  # Projection in the past works similar
  shortTrendExponent = sign(projectionPeriod) * pmin(abs(projectionPeriod), 10)
  longTrendExponent = sign(projectionPeriod) * pmax(abs(projectionPeriod) - 10, 0)

  cx = ((1 - object@trend)^(shortTrendExponent)) * ((1 - object@trend2)^(longTrendExponent))

  # Trend applies only fully to ages above 65, not up to 55, and linearly between
  (ages <= 55)*1 +
    (55 < ages & ages <= 65) * ((65 - ages) + (10 - (65 - ages)) * cx) / 10 +
    (ages > 65) * cx
}

mortalityTable.heubeck05 = setClass(
  "mortalityTable.heubeck05",
  slots = list(projectionFunction = "function"),
  prototype = list(projectionFunction = projection.Heubeck05),
  contains = "mortalityTable.trendProjection"
)

setMethod("periodDeathProbabilities", "mortalityTable.heubeck05",
          function(object,  ..., ages = NULL, Period = 2018) {
            qx = object@deathProbs * (1 + object@loading);
            factors = object@projectionFunction(object, Period - object@baseYear)
            finalqx = qx * factors

            fillAges(
              object@modification(finalqx),
              givenAges = ages(object),
              neededAges = ages)
          })


setMethod("deathProbabilities", "mortalityTable.heubeck05",
          function(object,  ..., ages = NULL, YOB = 1975) {

            # browser()
            qx = object@deathProbs * (1 + object@loading);
            givenAges = object@ages;
            factors = object@projectionFunction(object, YOB + givenAges - object@baseYear)
            finalqx = qx * factors

            fillAges(
              object@modification(finalqx),
              givenAges = ages(object),
              neededAges = ages)
          })



#---------------------------------------------------------------------------------------------- -
# Functions and classes to handle Heubeck's table at pensionTable-level                    ----
#---------------------------------------------------------------------------------------------- -

pensionTableProbArrange.Heubeck05 = function(x, q, i, qi, r, ap, api, qp, h, qw, yx, qg, as.data.frame = TRUE, table, ...) {
  # 1) Die obigen Ausscheidewahrscheinlichkeiten sind unabhänge Wahrscheinlichkeiten => Umrechnung auf abhängige
  i.stern = i
  q.stern = q
  i = i.stern * (1 - 1/2 * q.stern)
  q = q.stern * (1 - 1/2 * i.stern)


  # 2) Herleitung der Gesamtsterblickeit aus den anderen Wahrscheinlichkeiten
  #       -) Ausscheideordnungen:
  la = head(Reduce('*', (1 - q - i), init = 100000, accumulate = TRUE), -1)
  lg = la
  for (idx in seq_along(lg)) {
    qg[idx] = qi[idx] - la[idx]/lg[idx] * (qi[idx] - q[idx] - i[idx] * 1/2 * qi[idx] / (1 - 1/2*qi[idx]))
    lg[idx + 1] = lg[idx] * (1 - qg[idx])
  }
  lg = head(lg, -1)


  #       -) qg daraus bestimmen:
  qg = qi - la/lg * (qi - q - i * 1/2 * qi/(1 - 1/2*qi))


  # 3) Alterspensionistensterblichkeit ist qg bis Alter 73, danach normal extrapoliert
  qp[x < 74] = qg[x < 74]


  # TODO: Witwensterblichkeiten stimmen noch nicht!!!

  if (as.data.frame) {
    data.frame(x, q, i, qi, r, ap, api, qp, h, qw, yx, qg)
  } else {
    states = c("a", "i", "p", "d")
    transProb = array(0, dim = c(length(states), length(states), length(x)), dimnames = list(states, states, x))

    transProb["a", "a", ] = (1 - i - q) * (1 - ap);
    transProb["a", "i", ] = i;
    transProb["a", "p", ] = (1 - i - q) * ap;
    transProb["a", "d", ] = q;

    transProb["i", "a", ] = r;
    transProb["i", "i", ] = (1 - qi - r) * (1 - api);
    transProb["i", "p", ] = (1 - qi - r) * api;
    transProb["i", "d", ] = qi;

    transProb["p", "p", ] = 1 - qp;
    transProb["p", "d", ] = qp;

    transProb["d", "d", ] = 1;

    list(transitionProbabilities = transProb, widows = data.frame(x, h, qw, yx))
  }
}

pensionTable.Heubeck05 = setClass(
  "pensionTable.Heubeck05",
  slots = list(),
  prototype = list(
    probs.arrange = pensionTableProbArrange.Heubeck05
  ),
  contains = "pensionTable"
)







DAV2005G.generate = function(sex = "M") {
  table05 = function(name, data, trenddata, agevar = "x", probvar, dim = list(), ..., baseyear = 2005, projectionFunction = projection.Heubeck05) {
    if (is.null(trenddata)) {
      mortalityTable.period(
        name = name, ages = data[[agevar]], baseYear = baseyear,
        deathProbs = data[[probvar]], ...,
        data = list(dim = dim))
    } else {
      mortalityTable.heubeck05(
        name = name, ages = data[[agevar]], baseYear = baseyear,
        deathProbs = data[[probvar]],
        trend = trenddata$fk,
        trend2 = trenddata$fl,
        # dampingFunction = function(x) max(x, 0),
        data = list(dim = dim),
        projectionFunction = projectionFunction,
        ...)
    }
  }
  name05 = function(desc) {
    paste("DAV 2005-G, ", desc, sep = "")
  }

  basedata = utils::read.csv(
    system.file("extdata", paste0("1_Basis2005", sex), package = "PensionTablesHeubeck05"),
    # file.path("inst", "extdata", paste0("1_Basis2005", sex)),
    skip = 4,
    header = FALSE,
    col.names = c("Alter", "qgp", "qi", "qa", "i", "h", "xy", "qw"),
    dec = ",",
    sep = ";"
  )
  loadTrendDAV05 = function(sex) {
    utils::read.csv(
      system.file("extdata", paste0("1_Trend", sex), package = "PensionTablesHeubeck05"),
      # file.path("inst", "extdata", paste0("1_Trend", sex)),
      skip = 4,
      header = FALSE,
      col.names = c("Alter", "fk", "fl"),
      dec = ",",
      sep = ";"
    )
  }
  trenddata = loadTrendDAV05(sex)
  trenddata.Widow = loadTrendDAV05(sex = if (sex == "M") "W" else "M")

  # qa und ix in unabhängige Wahrscheinlichkeiten umrechnen
  basedata$qa = with(
    basedata,
    1/2*(2 + qa - i) - sqrt(1/4 * (2 + qa - i)^2 - 2 * qa)
  )
  basedata$i = with(basedata, i / (1 - 1/2 * qa))

  sex = tolower(sex)
  sexName = if (sex == "m") "Männer" else "Frauen"
  pensionTable.Heubeck05(
    name = name05(paste0("Pensionstafel ", sexName)),
    baseYear = 2005,
    qx =  table05(name05("qax, active males"),
                  basedata, trenddata,
                  "Alter", "qa",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Tod")),
    ix =  table05(name05("ix, probability of invalidity"),
                  basedata, NULL,
                  "Alter", "i",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Invalidisierung")),
    qgx = table05(name05("qgx, total males"),
                  basedata, trenddata,
                  "Alter", "qgp",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Tod")),
    qix = table05(name05("qix, disabled males"),
                  basedata, trenddata,
                  "Alter", "qi",
                  projectionFunction = projection.Heubeck05.qi,
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Tod")),
    rx =  mortalityTable.zeroes(name = "No reactivation", ages = basedata$Alter),
    apx = mortalityTable.onceAndFuture(transitionAge = 65 - 1, name = "Pensionsalter 65", ages = basedata$Alter),
    qpx = table05(name05("qpx, retired males"),
                  basedata, trenddata,
                  "Alter", "qgp",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Tod")),
    hx =  table05(name05("hx, marriage probability"),
                  basedata, NULL,
                  "Alter", "h",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Partnerwahrscheinlichkeit im Tod")),
    qwy = table05(name05("qwy, widows"),
                  basedata, trenddata.Widow,
                  "Alter", "qw",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "Tod")),
    yx =  table05(name05("y(x), age of widow"),
                  basedata, NULL,
                  "Alter", "xy",
                  dim = list(sex = sex,
                             collar = "Mischbestand",
                             type = "Pensionstafel Deutschland",
                             data = "official",
                             year = "DAV 2005-G",
                             risk = "mittl. Hinterbliebenenalter")),
    invalids.retire = FALSE,
    data = list(
      Geschlecht = sexName,
      Bestand = "Mischbestand",
      Invalidisierung = "Invaliditätspension"
    )
  )

}

DAV2005G.male = DAV2005G.generate("M")
DAV2005G.female = DAV2005G.generate("W")
DAV2005G = array(
  data = c("m" = DAV2005G.male, "w" = DAV2005G.female),
  dim = c(2,1),
  dimnames = list("Geschlecht" = c("m", "w"), "Bestand" = c("Mischbestand"))
)


# probs = transitionProbabilities(DAV2005G.male, YOB = 1955)
# # probs = transitionProbabilities(DAV2005G.female, YOB = 1955)
#
# View(round(probs,6))

rm(DAV2005G.generate)
