stopifnot(require(methods), require(utils), require(MortalityTables), require(dplyr), require(readxl)) # MortalityTable classes; new; Excel reader

# TODO:
#   - Save correction factor for ix (reactivation) inside the table
#   - Provide functions to adjust table for given pension age (correction factor for ix)
#   - Provide function to switch to qp at the pension age

############################################################################### #
# Pension Table AVÖ 2018-P                                                   ----
############################################################################### #

if (is.null(getOption("MortalityTables.AVOe2018PAng")) && is.null(getOption("MortalityTables.AVOe2018PMisch"))) {
  message("The pension table AVOe2018P requires the Excel files 'AVOe2018P_Pensionstafeln_Angestellte.xlsx' or 'AVOe2018P_Pensionstafeln_Mischbestand.xlsx'. Please set the paths to them with options(\"MortalityTables.AVOe2018PAng\" = \"/path/to/AVOe2018P_Pensionstafeln_Angestellte.xlsx\") and options(\"MortalityTables.AVOe2018PMisch\" = \"..\").")
} else {

    AVOe2018P = array(
        data = c(mortalityTable.NA),
        dim = c(3, 2, 2),
        dimnames = list(Geschlecht = c("m", "w", "u"), Bestand = c("Angestellte", "Mischbestand"), Invalidisierung = c("IP", "IPRG"))
    )

    #------------------------------------------------------------------------------- -
    # Functions to load the tables (raw base tables and trend) and the additional info ----
    #------------------------------------------------------------------------------- -

    AVOe2018P.trend.damping.fun = function(baseYear = 2008, eta = 1/200, zeta = 0) function(t){
      # 1/eta * atan(((t + baseYear) - 2016) * eta)
      1/eta * atan((t - zeta) * eta)
    }
    # AVOe2018P.trend.damping = AVOe2018P.trend.damping.fun()
    AVOe2018P.trend.damping = function(t) t



    loadAVOe2018P.data = function(file, sex = "M", collar = "Angestellte", IPtype = "IP") {
      if (is.null(file)) {
        return()
      }
      if (!sex %in% c("M", "F", "U")) {
        warning("Invalid sex given in call to loadAVOe2018P.table: ", sex, "; Allowed values are M, F and U.")
        return()
      }
      if (!collar %in% c("Angestellte", "Mischbestand")) {
        warning("Invalid collar given in call to loadAVOe2018P.table: ", collar, "; Allowed values are Angestellte and Mischbestand.")
        return()
      }
      if (!IPtype %in% c("IP", "IPRG")) {
        warning("Invalid type given in call to loadAVOe2018P.table: ", IPtype, "; Allowed values are IP and IPRG.")
        return()
      }

      sheet = recode(sex, "M" = "Männer", "F" = "Frauen", "U" = "Unisex")
      if (IPtype == "IPRG") {
        sheet = paste(sheet, "RG")
      }
      basedata = read_excel(
          file, sheet = sheet, skip = 2, n_max = 1, #col_names = FALSE,
          col_types = c("skip", "text", "skip", "skip", "text", "skip", "skip", "text", "skip", "skip", "numeric", "skip", "skip", "numeric"),
          col_names = c("sex", "collar", "IPtype", "baseyear", "trend")
          )

      baseYear = basedata$baseyear
      trendReduction = basedata$trend
      dataSex = basedata$sex
      dataCollar = basedata$collar
      dataIP = basedata$IPtype

      data = read_excel(
        file, sheet = sheet, skip = 5,
        col_names = c("Alter", "qa", "lambda.a", "qi", "lambda.i", "qg", "lambda.g", "qp", "lambda.p", "qw", "lambda.w", "ix", "hx", "yx")
      )
      # data = utils::read.csv(
      #   system.file("extdata", filename, package = "PensionTablesAVOe2018P"),
      #   skip = 4,
      #   encoding = "UTF-8",
      #   header = TRUE, check.names = FALSE,
      # )


      table18p = function(name, data, agevar = "x", probvar, trendvar = NULL, dim = list(), ..., baseYear = 2008, projectionFunction = projection.Heubeck05) {
        tbl = mortalityTable.period(
          name = name, ages = data[[agevar]], baseYear = baseYear,
          deathProbs = data[[probvar]],
          ...,
          data = list(dim = dim)
        )
        if (!is.null(trendvar)) {
          tbl = tbl %>%
            mT.addTrend(trend = data[[trendvar]], dampingFunction = AVOe2018P.trend.damping.fun(baseYear = baseYear, eta = trendReduction))
        }
        tbl
      }
      name18p = function(prob = "", sex = "m", collar = "Angestellte", IPtype = NULL) {
        if (!is.null(IPtype)) {
          IPtype = paste0(" (", IPtype, ")")
        } else {
          IPtype = ""
        }
        sprintf("AVÖ 2018-P %s%s %s %s",
                recode(prob,
                       "qx"   = "Aktivensterblichkeit",
                       "qix"  = "Invalidensterblichkeit",
                       "ix"   = "Invalidisierung",
                       "qpx"  = "Pensionistensterblichkeit",
                       "qgx"  = "Gesamtsterblichkeit",
                       "hx"   = "Partnerwahrscheinlichkeit im Tod",
                       "yx"   = "mittl. Hinterbliebenenalter",
                       "qwy"  = "Witwensterblichkeit",
                       "qwx"  = "Witwersterblichkeit",
                       "lambda"    = "Sterblichkeitstrend",
                       "lambda.wx" = "Sterblichkeitstrend Witwer",
                       "lambda.wy" = "Sterblichkeitstrend Witwen"
                ),
                IPtype,
                recode(sex, "m" = "Männer", "w" = "Frauen", "u" = "Unisex",
                       "M" = "Männer", "F" = "Frauen", "U" = "Unisex"),
                recode(collar, "Angestellte" = "Ang.", "Mischbestand" = "Mischb.", "Arbeiter" = "Arb.")
        )
      }


      defaultDims = list(
        table = "AVÖ 2018-P",
        type = "Pensionstafel Österreich",
        data = "official",
        year = "2018",
        sex = sex,
        collar = collar,
        IPtype = IPtype
      )

      pensionTable(
        name = name18p("Pensionstafel", collar = collar, sex = sex, IPtype = IPtype),
        baseYear = baseYear,
        qx = table18p(
          name18p("qx", sex, collar, IPtype),
          data, "Alter", "qa", "lambda.a",
          dim = c(defaultDims, list(
            probability = "qx",
            risk = "Tod"
          ))),
        ix = table18p(
          name18p("ix", sex, collar, IPtype),
          data, "Alter", "ix", NULL,
          dim = c(defaultDims, list(
            probability = "ix",
            risk = "Invalidisierung"
          ))),
        qgx = table18p(
          name18p("qgx", sex, collar, IPtype),
          data, "Alter", "qg", "lambda.g",
          dim = c(defaultDims, list(
            probability = "qgx",
            risk = "Tod"
          ))),
        qix = table18p(
          name18p("qix", sex, collar, IPtype),
          data, "Alter", "qi", "lambda.i",
          dim = c(defaultDims, list(
            probability = "qix",
            risk = "Tod"
          ))),
        rx =  mortalityTable.zeroes(name = "No reactivation", ages = data$Alter),
        apx = mortalityTable.onceAndFuture(transitionAge = 65 - 1, name = "Pensionsalter 65", ages = data$Alter),
        qpx = table18p(
          name18p("qpx", sex, collar, IPtype),
          data, "Alter", "qp", "lambda.p",
          dim = c(defaultDims, list(
            probability = "qpx",
            risk = "Tod"
          ))),
        hx = table18p(
          name18p("hx", sex, collar, IPtype),
          data, "Alter", "hx", NULL,
          dim = c(defaultDims, list(
            probability = "hx",
            risk = "Partnerwahrscheinlichkeit im Tod"
          ))),
        qwy = table18p(
          name18p("qwy", sex, collar, IPtype),
          data, "Alter", "qw", "lambda.w",
          dim = c(defaultDims, list(
            probability = "qwy",
            risk = "Tod"
          ))),
        yx = table18p(
          name18p("yx", sex, collar, IPtype),
          data, "Alter", "yx", NULL,
          dim = c(defaultDims, list(
            probability = "yx",
            risk = "mittl. Hinterbliebenenalter"
          ))),

        invalids.retire = FALSE,
        data = list(
          Geschlecht = sex,
          Bestand = collar,
          Invalidisierung = IPtype
        )
      )
    }


    for (sex in dimnames(AVOe2018P)$Geschlecht) {
      for (collar in dimnames(AVOe2018P)$Bestand) {
        for (IPtype in dimnames(AVOe2018P)$Invalidisierung) {
          AVOe2018P[[sex, collar, IPtype]] = loadAVOe2018P.data(
            file = if(collar == "Mischbestand")
              getOption("MortalityTables.AVOe2018PMisch")
            else
              getOption("MortalityTables.AVOe2018PAng"),
            sex = recode(sex, "m" = "M", "w" = "F", "u" = "U"),
            collar = collar,
            IPtype = IPtype)
        }
      }
    }

    rm(loadAVOe2018P.data)
}
